package edu.cmu.cs.cs214.blockingqueue;


import java.util.List;
import java.util.ArrayList;
import java.util.Collections;

/**
 * This class implements a bounded BlockingQueue.  
 *
 * Like any basic queue, the BlockingQueue has enqueue and dequeue 
 * methods.  The BlockingQueue is bounded, which means the queue has a 
 * maximum number of elements it can hold.  The idea of a BlockingQueue 
 * is as follows:
 *   1. The BlockingQueue should be thread-safe; i.e., multiple threads 
 *      can safely use a single BlockingQueue without externally 
 *      synchronizing their actions, and the BlockingQueue will not have 
 *      any internal race conditions.
 *   2. If a thread calls dequeue on an empty queue, that thread will
 *      block (i.e., wait) until another thread enqueues an element.
 *   3. If a thread enqueues an element into a full queue, that thread 
 *      will block until another thread dequeues (which makes room for 
 *      the new element).
 *
 * Note that enqueue and dequeue on a BlockingQueue will always eventually
 * succeed, as long as another thread eventually dequeues (to make space)
 * or enqueues (to provide an element to dequeue), respectively.
 * 
 * In our representation the head index refers to the item at the front of 
 * the queue and the tail index refers to the item at the end of the queue.
 * Our queue representation prevents null elements from being inserted
 * into the queue and always resets the cell to null when an item is
 * dequeued.  The head and tail items are guaranteed to be non-null if
 * the queue is non-empty, and will always be null in a (properly
 * implemented) empty queue.  To check if there is space for a new item
 * in a properly implemented queue, you can check if the space after the
 * tail index is null.  
 *
 * @param <E> - The type of the elements in the BlockingQueue
 */
public class ArrayBlockingQueue<E> implements BlockingQueue<E> {
    final List<E> list;
    final int capacity;
    int head, tail;
    
    final Object lock;
    
    /**
     * Constructs a BlockingQueue with the specified capacity.
     * 
     * @param capacity The capacity of the BlockingQueue.
     */
    public ArrayBlockingQueue(int capacity) {
      if (capacity <= 0) {
          throw new IllegalArgumentException("Capacity must be positive");
      }
      // Create a new list of the correct size with all elements set to null.
      this.list = new ArrayList<E>(Collections.nCopies(capacity, (E)null));
      this.head = 0;
      this.tail = 0;
      this.capacity = capacity;
      this.lock = new Object();
    }

    /** 
     * Constructs a BlockingQueue with default capacity 10.
     */
    public ArrayBlockingQueue() {
      this(10);
    }

    /* (non-Javadoc)
	 * @see BlockingQueue#enqueue(E)
	 */
    public void enqueue(E e) {
        if (e == null) {
            throw new IllegalArgumentException("Can't enqueue null");
        }
        synchronized(lock) {
            while (list.get((tail+1)%capacity) != null) {  // Queue is full iff tail+1 != null.
                try {
                    lock.wait();  // Blocks here until interrupted or notified.
                } catch (InterruptedException ignore) { /* ignore */ }
            }
            list.set(tail, e);
            tail = (tail+1) % capacity;
            lock.notifyAll();
        }
    }
    
    /* (non-Javadoc)
	 * @see BlockingQueue#dequeue()
	 */
    public E dequeue() {
    	synchronized(lock) {
            while (list.get(head) == null) {  // The head is null iff queue is empty.
                try {
                    lock.wait();              // Blocks here until interrupted or notified.
                } catch (InterruptedException ignore) { /* ignore */ }
            }
            E rv = list.get(head);
            list.set(head, null);
            head = (head+1) % capacity;
            lock.notifyAll();
            return rv;
        }
    }
    
    public static void main(String[] args) {
    	// Starts new threads for a producer and a consumer to access a 
    	// single queue.
        ArrayBlockingQueue<Integer> queue = new ArrayBlockingQueue<Integer>();
        Producer p = new Producer(queue);
        Consumer c = new Consumer(queue);
        new Thread(p).start();
        new Thread(c).start();
    }
}

